/////////////////////////////////////////////////////////////////////////////////
//
// This is a special purpose shader that is used to add a "fairy magic" effect
// to VGHD clip. It is based on Emmanuel Keller's "Magic Particles" shader and
// the work done for this version fed back into the  "Magic Particles"  shader
// that you can find in the ShaderToy adaptions directory.
//
// This shader is intended to be applied to the images corresponding to frames
// of an animation and adds an overlay of "magical sparkles".  The screen area
// of each clip will be relatively small so each use of the shader should only
// place a relatively small load on the GPU.  However, a single scene may have
// several clips using the shader and the overall load may then be high and if
// so may result in the final image stuttering. If this occurs the load may be
// lowered by reducing the number of particles generated by changing the value
// of SECONDARY_PARTICLE_COUNT to something smaller - see below for details.
//
// Note, as well as using slightly modified parameters and a main program that
// applies the "magic particles" effect to what is assumed to be a VGHD clip's
// image this version of the shader differs from that found in  the  ShaderToy
// adaptions by suppressing the display of the main particle. This suppression
// is done because we want the sparkles to seem to be spontaneously  generated
// by the girls and not by something else.
//
/////////////////////////////////////////////////////////////////////////////////

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed

uniform sampler2D SourceImage; // The image that is to be modified
varying vec4 gl_TexCoord[];    // Normalised position within image

/////////////////////////////////////////////////////////////////////////////////
//
// When this shader is used as intended SourceImage corresponds to the image of
// a VGHD clip and gl_TexCoord[0] will give the coordinates within that texture
// corresponding to the screen pixel currently being processed.  The cordinates
// are normalised so (0.0,0.0) and (1.0,1.0) correspond to opposite corners  of
// the image. This is very convenient as you can simply use
//
//      vec4 color = texture2D ( SourceImage, gl_TexCoord[0].xy );
//
// to get the colour, including the opacity, to use for the pixel and  if  this
// is assigned to gl_FragColor the clip is displayed with no modifications, but
// the shader can also make changes to alter the appearence of the clip.
//
// However, a some considerations need to be taken into account when using this
// data, these being
//
//    (0,0) is nominaly the top left  corner  of  the  image  and  (1,1)
//    the bottom right corner. These may however be  reversed  depending
//    on scale and rot factors for the clipSprite in the scene file.
//
//    The full texture corresponding to the clipSprite includes a  large
//    transparent region surrounding the image of the performer but this
//    is clipped to the smallest rectangle to minimise processing of any
//    pixels that are known to be transparent.
//
// As a result of the image clipping the range of the texture co-ordinates seen
// by the shader will be less than the nominal 0.0 to 1.0 range.  This can be a
// problem when using a shader to mofify a clip as any  modifications  will  be
// restricted to the clipped image area and so be cut off at the edges of  that
// area making that edge be visible, sometimes very visible, e.g. as a coloured
// box surrounding the performer.
//
// Furthermore, I currently do not know of any way to determine where the edges
// of the clipped image are so I know of no way to avoid the problem completely
// other than never modifying transparent pixels.  However,  by careful design,
// the effect can be minimised.  This is the case with the current shader which
// overlays the image with a set of moving paticles. Although any particle that
// moves outside of the clipped area will disappear as it crosses the  boundary
// revealing the existance of the boundary as it crosses it - but the particles
// are small and cross the boundary quickly.
//
/////////////////////////////////////////////////////////////////////////////////

/*

"Magic particles" by Emmanuel Keller aka Tambako - December 2015
License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
Contact: tamby@tambako.ch

Modified by TheEmu, Jan 2016.

   Removed background
   Minor optimisations
   Inlining most functions
   Added variable transparency
   Comment changes for clarity
   Some name changes for clarity
   Added main configuration options
   Increased secondary particle size
   Increased particle colour saturation
   
   Parameters changed to suit "Naughty Fairies" scenes

*/

/////////////////////////////////////////////////////////////////////////////////
//
// The basic model is that of a main particle randomly wandering within a fixed
// rectangular area and throwing off secondary particles that not  only  wander
// randomly but also drift in some direction as if they are falling or floating
// upwards. The main particle is long lived and does not fade but the secondary
// particles fade and die but are then replaced by more secondaries. All of the
// particles vary both their colour and intensity with time which, depending on
// its frequency, may appear as pulsation or flicker. The shape of the main and
// secondary particles may be separately configured to be blobs or stars.
//
/////////////////////////////////////////////////////////////////////////////////

// Main configuration options. These may be used to adapt the shader for some
// particular purpose or to reduce the load that it places on the GPU.

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// The shader generates SECONDARY_PARTICLE_COUNT sets of secondary particles
// with each set comprising SECONDARY_PARTICLE_MULTIPLICITY radiating from a
// common origin. SECONDARY_PARTICLE_COUNT must be defined and may have  any
// integer value, though values less than 1 inhibit all secondary particles.
// SECONDARY_PARTICLE_MULTIPLICITY is required  if  SECONDARY_PARTICLE_COUNT
// is greater than 0 and may only be defined as one of 1, 2, 4 or 8.

   #define SECONDARY_PARTICLE_COUNT         8
   #define SECONDARY_PARTICLE_MULTIPLICITY  4

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// If SECONDARY_PARTICLE_MULTIPLICITY is 2 one but not both of the following
// symbols may be defined to modify the manner in which the pair  move  away
// from their common point of origin. If neither is defined then they travel
// opposite directions but if one of DUAL_PARTICLES_X or DUAL_PARTICLES_Y is 
// only the X or Y components of their delta velocities are reversed.

// #define DUAL_PARTICLES_X
// #define DUAL_PARTICLES_Y

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// The secondary particles may take the form of spherical blobs or stars with
// either 4 or 8 points depending on which, if any,  of the following symbols
// have been defined.  If SECONDARY_ORTHOGONAL_RAYS is defined then secondary
// particles will have rays in the vertical and horizontal directions and  if
// SECONDARY_DIAGONAL_RAYS is defined then secondary particles will have rays
// in the diagonal direction. If both are defines then both sets of rays will
// be generated giving a eight pointed star while if neither are defined then
// only a central spherical blob will be generated.  Although  eight  pointed
// stars are the best looking option for most purposes reducing them to  four
// pointed stars or even to just their central blobs reduces the load on  the
// GPU significantly so it may be preferable to reduce the number of rays  to
// 4, or even to 0, rather than reduce SECONDARY_PARTICLE_COUNT.

   #define SECONDARY_ORTHOGONAL_RAYS
   #define SECONDARY_DIAGONAL_RAYS

// The secondary particle may optionaly either sparkle or just fade away.  If
// sparkling is enabled by defining SECONDARIES_SPARKLE then the  intensities
// of the secondary particles will vary with the variation  being  controlled
// by the spakling control constants. If SECONDARIES_SPARKLE is  not  defined
// then the secondary particles will simply fade with time.  If  sparkling is
// inhibited then the average particle intensity will be lower than if it  is
// enabled and it is therefore probable that INTENSITY_FACTOR will need to be
// increased to compensate.

   #define SECONDARIES_SPARKLE

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Options controling the movement of the main and of the secondary particles
// relative to it.  If MOVING_MAIN_PARTICLE is defined then the main particle
// will move in accordance with the amp, freq and phase  harmonic  parameters
// and the seconadry particles move with it. If ENABLE_DRIFT_FORCE is defined
// then the effect of the drift force on the secondary particles  is  enabled
// and they drift in the direction of the drift vector.  If ENABLE_SCATTER is
// is defined then he secondary particles will scatter with random velocities
// from their initial positions.

   #define MOVING_MAIN_PARTICLE
   #define ENABLE_DRIFT_FORCE
   #define ENABLE_SCATTER

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// As with the secondary particles the main particle may also take  the  form
// of a blob or a star with 4 or 8 points. This is controled by the following
// symbols in the same way as that for the secondary particles except that if
// MAIN_ORTHOGONAL_RAYS and both MAIN_DIAGONAL_RAYS are defined then no  main
// particle is generated unless SHOW_MAIN_PARTICLE is defined. For the "Fairy
// Magic" shader we suppress the display of the main particle.

// #define MAIN_ORTHOGONAL_RAYS
// #define MAIN_DIAGONAL_RAYS
// #define SHOW_MAIN_PARTICLE

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// The overall intensity (and hence visible size) of the blobs may be  adjusted
// by changing the value specified by the following parameter.  A  value of 1.0
// seems to be suitable for eight pointed stars but may be a little too low for
// the other options. Typical values will be in the range 0.2 to 5.0 but values
// outside this range are valid and can be used. If sparkling is inhibited, see
// above, then INTENSITY_FACTOR should be increased to compensate.

   #define INTENSITY_FACTOR 1.0

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// If MOTION_BLUR is defined then motion blur is modeled, however this places a
// heavy load on the GPU as it entails repeating everything MOTION_BLUR times.

// #define MOTION_BLUR 2

/////////////////////////////////////////////////////////////////////////////////

// If either MAIN_ORTHOGONAL_RAYS or MAIN_DIAGONAL_RAYS is defined then ensure
// that SHOW_MAIN_PARTICLE is also defined so that the main particle is seen.

#ifndef SHOW_MAIN_PARTICLE
  #if defined(MAIN_ORTHOGONAL_RAYS) || defined(MAIN_DIAGONAL_RAYS)
     #define SHOW_MAIN_PARTICLE
  #endif
#endif

// Ensure INTENSITY_FACTOR is defined.

#ifndef INTENSITY_FACTOR
   #define INTENSITY_FACTOR 1.0
#endif

/////////////////////////////////////////////////////////////////////////////////

// A few mathematical constants.

#define tau       ( 6.283185307179586476925286766559 ) // 2 * pi
#define d2r       ( 0.017453292519943295769236907685 ) // tau / 360
#define sqrt_2    ( 1.414213562373095048801688724210 ) // sqrt(2.0)
#define sqrt_half ( 0.707106781186547524400844362105 ) // sqrt(0.5)

/////////////////////////////////////////////////////////////////////////////////

// Secondary configuration options. These parameters may be adjusted to fine tune
// the shader for a particular use.  They are much more specific in their effects
// than the main configuration options that have been described above.

// General particles constants.

const vec2 gen_scale = vec2(1.00,1.00);        // Scale factor for positions.
const vec2 mid_point = vec2(0.00,0.00);        // Position offset of the particles.

// Main particle movement constants.

#ifdef MOVING_MAIN_PARTICLE
const vec3 x_amp   = vec3(0.60,0.15,0.08)*0.2; // Amplitudes of the harmonics of horizontal position.
const vec3 y_amp   = vec3(0.30,0.20,0.12)*0.3; // Amplitudes of the harmonics of vertical position.
const vec3 x_freq  = vec3(0.19,0.16,0.19)*tau; // Frequencies (in radians/sec) of horizontal position.
const vec3 y_freq  = vec3(0.11,0.14,0.21)*tau; // Frequencies (in radians/sec) of vertical position.
const vec3 x_phase = vec3(0.00,45.0,55.0)*d2r; // Phases (in radians) of the horizontal position.
const vec3 y_phase = vec3(90.0,120.,10.0)*d2r; // Phases (in radians) of the vertical position.
#endif

// Secondary particle movement constants.

const vec2  drift_force  = vec2(0.00,-0.02);   // Gravitation, buoyancy, wind or other force vector.
const float min_scatter  = 0.04;               // Min movement out of the trajectory in display units/sec.
const float max_scatter  = 0.25;               // Max movement out of the trajectory in display units/sec.
const float timefact_min = 6.00;               // Min time particle moves slower than the main particle.
const float timefact_max = 20.0;               // Max time particle moves slower than the main particle.

// Particle time constants.

const float speed_factor  = 1.00;              // Time factor, < 1.0 slow motion, > 1.0 for fast.
const float start_time    = 19.9;              // Time in seconds until all particles are launched.
const float life_time_min = 4.00;              // Minimum life time of a particle in seconds.
const float life_time_max = 4.50;              // Maximum life time of a particle in seconds.
const float growth_time   = 0.15;              // Fraction of lifetime to reach maximum intensity.

// Particle intensity constants.

const float min_intensity_factor  = 0.50;      // Minimum initial intensity factor.
const float max_intensity_factor  = 4.40;      // Maximum initial intensity factor.
const float mp_initial_intensity  = 6.00;      // Initial intensity of the main particle.
const float distance_factor       = 3.00;      // Distance factor applied before calculating intensity.
const float intensity_exponent    = 2.30;      // Exponent of the intensity in function of the distance.
const float mp_intensity_exponent = 2.30;      // Exponent of the intensity in function of the distance.

// Secondary particle sparkling constants.

#ifdef SECONDARIES_SPARKLE
const float min_spark_intensity   = 2.50;      // Minimum sparkling intensity factor.
const float max_spark_intensity   = 6.00;      // Minimum sparkling intensity factor.
const float min_spark_frequency   = 4.50*tau;  // Minimum sparkling frequence in radians/sec.
const float max_spark_frequency   = 6.00*tau;  // Maximum sparkling frequence in radians/sec.
const float spark_freq_time_fact  = 0.35;      // Sparkling frequency factor at end of a life.
#endif

// Particle color constants.

const float min_hue_shift     = -0.200;        // Minimum particle hue shift (spectrum width = 1.0)
const float max_hue_shift     =  0.800;        // Maximum particle hue shift (spectrum width = 1.0)
const float min_saturation    =  0.500;        // Minimum particle saturation (0.0 to 1.0)
const float max_saturation    =  0.900;        // Maximum particle saturation (0.0 to 1.0)
const float saturation_factor =  0.450;        // Particle saturation factor
const float hue_time_factor   =  0.035;        // Time-based hue shift.
const float mp_hue_shift      =  0.500;        // Hue shift of the main particle.
const float mp_sat_shift      =  0.180;        // Saturation delta of the main particle.
const float mp_sat_pow1       = -2.500;        // Main particle saturation exponent.
const float mp_sat_pow2       =  0.075;        // Main particle saturation factor.

// Particle star constants.

const vec2  star_hv_dfac   = vec2(9.0,0.32);   // x-y transformation vector for orthogonal rays.
const vec2  star_diag_dfac = vec2(13.,0.61);   // x-y transformation vector for diagonal rays.
const float star_hv_ifac   = 0.25;             // Intensity factor of the orthogonal rays.
const float star_diag_ifac = 0.19;             // Intensity factor of the diagonal rays.

// Particle motion blur constants.

#ifdef MOTION_BLUR
  const float blur_time = 0.02;                // Time shift of the motion blur in seconds
#endif

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Derived constants

const float particle_intensity_factor = INTENSITY_FACTOR / 40000.0;

const float mp_intensity_factors = particle_intensity_factor
                                 * max_intensity_factor
                                 * mp_initial_intensity;

const float inverse_distance_factor = 1.0 / distance_factor;
const float star_hv_dist_ifactor    = inverse_distance_factor * star_hv_ifac;
const float stard_dist_ifactor      = inverse_distance_factor * star_diag_ifac * sqrt_half;
const float dist_modifier_radial    = inverse_distance_factor * 0.020;
const float dist_modifier_hv        = inverse_distance_factor * 0.010;
const float dist_modifier_diag      = inverse_distance_factor * 0.010 * sqrt_half;

const vec2 scaled_mid_point = mid_point * gen_scale;

#ifdef MOVING_MAIN_PARTICLE
  const vec3 scaled_x_amp = x_amp * gen_scale.x;
  const vec3 scaled_y_amp = y_amp * gen_scale.y;
#endif

#ifdef ENABLE_SCATTER
  const vec2 scaled_min_scatter = min_scatter * gen_scale;
  const vec2 scaled_max_scatter = max_scatter * gen_scale;
#endif

#ifdef ENABLE_DRIFT_FORCE
  const vec2 scaled_drift_force = drift_force * gen_scale;
#endif

#ifdef SECONDARIES_SPARKLE
   const float max_spark_frequency_range = max_spark_frequency - min_spark_frequency;
#endif

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// HSV to RGB colour conversion from https://www.shadertoy.com/view/ldtGDn
// All instances have been expanded in line so we do not need the function
// itself, but it has been retained as a comment for reference. TheEmu.

const vec3 hsv2rgbXX = vec3(0.0,2.0/3.0,1.0/3.0) * tau;

// vec3 hsv2rgb ( vec3 hsv )
//  { hsv.yz = clamp ( hsv.yz, 0.0, 1.0 );
//    return hsv.z * ( hsv.y * ( cos ( hsv.x*tau + hsv2rgbXX ) - 1.0 ) + 1.0 );
//  }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// A simple random value generator. Generates values in the range 0.0 to 1.0. As
// a random number generator it is probably pretty poor, but we do not need high
// quality randomness for the purposes of this shader so a quick one is used.

// #define random(x) fract ( sin(x*12.989) * 43758.545 )
// #define random(x) fract ( sin(x) * 1254.538647 )
   #define random(x) fract ( (x) * 7.1880827289760327020821943451248)

// Independant random numbers. The values of the constants used in these  should
// not matter as long as the expressions remain distinct. The vector dot product
// is used as a efficient way to sum a set of products. Using  the  dot function
// instead of explicitly writing a*b+c*d+e*f is recomended in  GPU  optimisation
// guides as GPUs typically contain special purpose, and fast, hardware for this
// operation and although any decent optimising compiler ought to use it for the
// a*b+c*d+e*f form some compilers may not always do it.

#define random_A(kk)    random ( kk )
#define random_B(kk)    random ( kk*37.0 - 35.12345 )
#define random_C(kk,rn) random ( dot ( vec3(kk,rn,1.0), vec3( 3.0, 31.1,  1.31) ) )
#define random_D(kk,rn) random ( dot ( vec3(kk,rn,1.0), vec3( 5.0, 23.2,  3.23) ) )
#define random_E(kk,rn) random ( dot ( vec3(kk,rn,1.0), vec3( 7.0, 19.3,  5.19) ) )
#define random_F(kk,rn) random ( dot ( vec3(kk,rn,1.0), vec3(13.0, 17.4,  7.17) ) )
#define random_G(kk,rn) random ( dot ( vec3(kk,rn,1.0), vec3(17.0, 13.5, 11.13) ) )
#define random_H(kk,rn) random ( dot ( vec3(kk,rn,1.0), vec3(19.0, 11.6, 13.11) ) )
#define random_I(kk,rn) random ( dot ( vec3(kk,rn,1.0), vec3(23.0,  9.7, 17.07) ) )
#define random_J(kk,rn) random ( dot ( vec3(kk,rn,1.0), vec3(31.0,  7.8, 23.05) ) )

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Macros to support different particle multiplicities. The  action of a DO_Xn
// macro depends on the SECONDARY_PARTICLE_MULTIPLICITY.  If  SECONDARY_PARTICLE_MULTIPLICITY <= n
// then its sole argument will be used as part of the shader otherwise it will
// not. DO_X1 is not required but is used to maintain regularity..

#if SECONDARY_PARTICLE_COUNT > 0

   #if SECONDARY_PARTICLE_MULTIPLICITY == 8
   
     #define PVEC vec4
   
     #define DO_X1(stmt) stmt
     #define DO_X2(stmt) stmt
     #define DO_X4(stmt) stmt
     #define DO_X8(stmt) stmt
   
     const mat2 rot_p45 = mat2(sqrt_half,sqrt_half,-sqrt_half,sqrt_half);
     const mat2 rot_m45 = mat2(sqrt_half,-sqrt_half,sqrt_half,sqrt_half);
   
   #elif SECONDARY_PARTICLE_MULTIPLICITY == 4
   
     #define PVEC vec4
   
     #define DO_X1(stmt) stmt
     #define DO_X2(stmt) stmt
     #define DO_X4(stmt) stmt
     #define DO_X8(stmt)
   
   #elif SECONDARY_PARTICLE_MULTIPLICITY == 2
   
     #define PVEC vec2
   
     #define DO_X1(stmt) stmt
     #define DO_X2(stmt) stmt
     #define DO_X4(stmt)
     #define DO_X8(stmt)
   
   #elif SECONDARY_PARTICLE_MULTIPLICITY == 1
   
     #define PVEC float
   
     #define DO_X1(stmt) stmt
     #define DO_X2(stmt)
     #define DO_X4(stmt)
     #define DO_X8(stmt)
   
   #else
   
     #error Invalid, or missing, particle multiplicity.
   
   #endif

   const float max_scatter_angle = tau / float(SECONDARY_PARTICLE_MULTIPLICITY);

#endif

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Harmonic calculation.

#define harms(amp,freq,phase,time) dot ( amp, cos ( time*freq + phase ) )

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Main function to draw particles, outputs the rgb color.

vec4 drawParticles ( vec2 uv, float time_offset )
 {
   vec4 pcol = vec4(0.0);

   float time2 = speed_factor*iGlobalTime + time_offset;

   #if SECONDARY_PARTICLE_COUNT > 0

      // Generate secondary particles.

      float n_secondary = float(SECONDARY_PARTICLE_COUNT);

      for ( float i=n_secondary; i>0.0; i-- )
       {
         // Get all relevent times for the i'th secondary particle.

         float plt = mix ( life_time_min, life_time_max, random_B(i) );

         float pst = start_time * random_A(i); // Particle start time
         float tt  = time2 - pst;              // Time since the particle was first born.

         float run_nr = floor ( tt / plt );    // Number of the "life" of the particle.
         float time3  = mod   ( tt,  plt );    // Time within this life of the particle.
         float time3f = time3 / plt;           // Fraction of current lifetime.

         // Particle's "local" time, whenever a particle is "reborn" its local
         // time resets to 0. Note, it does not run at the normal time rate so
         // the motion calculated using it differs from what would be obtained
         // if a normal linear time was used.

         float t = mix ( timefact_min, timefact_max, random_C(i,run_nr) );
         float ptime = ( ( run_nr*plt + pst ) * ( t - 1.0 ) + time2 ) / t;

         // Initial intensity, growth and fading with time.

         float pint0 = mix ( min_intensity_factor, max_intensity_factor, random_F(i,run_nr) )
                     * (1.0-time3f) * particle_intensity_factor
                     * smoothstep ( 0.0, growth_time, time3f );

         // "Sparkling" of the particles.

         #ifdef SECONDARIES_SPARKLE

            float sparkfreq = min ( spark_freq_time_fact*time3, 1.0 ) * min_spark_frequency
                            + random_G(i,run_nr) * max_spark_frequency_range;

            pint0 *= mix ( min_spark_intensity, max_spark_intensity, random_H(i,run_nr) )
                   * ( sin(sparkfreq*time2)*0.5 + 1.0 );

         #endif

         // Get the particle's position. See above with regard to ptime.

         vec2 ppos = scaled_mid_point - uv;
         vec2 delta = vec2(0.0,0.0);

         #ifdef MOVING_MAIN_PARTICLE
            ppos += vec2 ( harms ( scaled_x_amp, x_freq, x_phase, ptime ),
                           harms ( scaled_y_amp, y_freq, y_phase, ptime )
                         );
         #endif

         #ifdef ENABLE_SCATTER
            float scatter_angle = random_D(i,run_nr) * max_scatter_angle;
            vec2  scatter_speed = mix ( scaled_min_scatter, scaled_max_scatter, random_E(i,run_nr) );
            delta += vec2(cos(scatter_angle),sin(scatter_angle)) * scatter_speed * time3;
         #endif

         #ifdef ENABLE_DRIFT_FORCE
            ppos += time3*time3 * scaled_drift_force;
         #endif

         DO_X1 ( vec2 ppos_a = ppos + delta; )

         #ifdef DUAL_PARTICLES_X
            DO_X2 ( vec2 ppos_b = ppos - delta*vec2(1.0,-1.0); )
         #elif defined(DUAL_PARTICLES_Y)
            DO_X2 ( vec2 ppos_b = ppos + delta*vec2(1.0,-1.0); )
         #else
            DO_X2 ( vec2 ppos_b = ppos - delta; )
         #endif

         DO_X4 ( vec2 ppos_c = ppos + delta.yx*vec2(1.0,-1.0); )
         DO_X4 ( vec2 ppos_d = ppos - delta.yx*vec2(1.0,-1.0); )

         DO_X8 ( vec2 ppos_e = ppos + delta*rot_p45; )
         DO_X8 ( vec2 ppos_f = ppos + delta*rot_m45; )
         DO_X8 ( vec2 ppos_g = ppos - delta*rot_p45; )
         DO_X8 ( vec2 ppos_h = ppos - delta*rot_m45; )

         // Generate either a blob or a four or eight pointed star.

         DO_X1 ( PVEC pint_ad = PVEC(0.0); )
         DO_X8 ( PVEC pint_eh = PVEC(0.0); )

         DO_X1 ( pint_ad.x = inverse_distance_factor / ( length(ppos_a) + dist_modifier_radial ); )
         DO_X2 ( pint_ad.y = inverse_distance_factor / ( length(ppos_b) + dist_modifier_radial ); )
         DO_X4 ( pint_ad.z = inverse_distance_factor / ( length(ppos_c) + dist_modifier_radial ); )
         DO_X4 ( pint_ad.w = inverse_distance_factor / ( length(ppos_d) + dist_modifier_radial ); )

         DO_X8 ( pint_eh.x = inverse_distance_factor / ( length(ppos_e) + dist_modifier_radial ); )
         DO_X8 ( pint_eh.y = inverse_distance_factor / ( length(ppos_f) + dist_modifier_radial ); )
         DO_X8 ( pint_eh.z = inverse_distance_factor / ( length(ppos_g) + dist_modifier_radial ); )
         DO_X8 ( pint_eh.w = inverse_distance_factor / ( length(ppos_h) + dist_modifier_radial ); )

         #ifdef SECONDARY_DIAGONAL_RAYS

            DO_X1 ( vec2 pposd_a = vec2 ( ppos_a.x+ppos_a.y, ppos_a.x-ppos_a.y ); )
            DO_X2 ( vec2 pposd_b = vec2 ( ppos_b.x+ppos_b.y, ppos_b.x-ppos_b.y ); )
            DO_X4 ( vec2 pposd_c = vec2 ( ppos_c.x+ppos_c.y, ppos_c.x-ppos_c.y ); )
            DO_X4 ( vec2 pposd_d = vec2 ( ppos_d.x+ppos_d.y, ppos_d.x-ppos_d.y ); )

            DO_X8 ( vec2 pposd_e = vec2 ( ppos_e.x+ppos_e.y, ppos_e.x-ppos_e.y ); )
            DO_X8 ( vec2 pposd_f = vec2 ( ppos_f.x+ppos_f.y, ppos_f.x-ppos_f.y ); )
            DO_X8 ( vec2 pposd_g = vec2 ( ppos_g.x+ppos_g.y, ppos_g.x-ppos_g.y ); )
            DO_X8 ( vec2 pposd_h = vec2 ( ppos_h.x+ppos_h.y, ppos_h.x-ppos_h.y ); )

            DO_X1 ( PVEC dist_abcd_d1 ); DO_X1 ( PVEC dist_abcd_d2 );
            DO_X8 ( PVEC dist_efgh_d1 ); DO_X8 ( PVEC dist_efgh_d2 );

            DO_X1 ( dist_abcd_d1.x = length ( pposd_a*star_diag_dfac.xy ); )
            DO_X1 ( dist_abcd_d2.x = length ( pposd_a*star_diag_dfac.yx ); )

            DO_X2 ( dist_abcd_d1.y = length ( pposd_b*star_diag_dfac.xy ); )
            DO_X2 ( dist_abcd_d2.y = length ( pposd_b*star_diag_dfac.yx ); )

            DO_X4 ( dist_abcd_d1.z = length ( pposd_c*star_diag_dfac.xy ); )
            DO_X4 ( dist_abcd_d2.z = length ( pposd_c*star_diag_dfac.yx ); )

            DO_X4 ( dist_abcd_d1.w = length ( pposd_d*star_diag_dfac.xy ); )
            DO_X4 ( dist_abcd_d2.w = length ( pposd_d*star_diag_dfac.yx ); )

            DO_X8 ( dist_efgh_d1.x = length ( pposd_e*star_diag_dfac.xy ); )
            DO_X8 ( dist_efgh_d2.x = length ( pposd_e*star_diag_dfac.yx ); )

            DO_X8 ( dist_efgh_d1.y = length ( pposd_f*star_diag_dfac.xy ); )
            DO_X8 ( dist_efgh_d2.y = length ( pposd_f*star_diag_dfac.yx ); )

            DO_X8 ( dist_efgh_d1.z = length ( pposd_g*star_diag_dfac.xy ); )
            DO_X8 ( dist_efgh_d2.z = length ( pposd_g*star_diag_dfac.yx ); )

            DO_X8 ( dist_efgh_d1.w = length ( pposd_h*star_diag_dfac.xy ); )
            DO_X8 ( dist_efgh_d2.w = length ( pposd_h*star_diag_dfac.yx ); )

            DO_X1 ( pint_ad += stard_dist_ifactor / ( dist_abcd_d1 + dist_modifier_diag )
                             + stard_dist_ifactor / ( dist_abcd_d2 + dist_modifier_diag );
                  )

            DO_X8 ( pint_eh += stard_dist_ifactor / ( dist_efgh_d1 + dist_modifier_diag )
                             + stard_dist_ifactor / ( dist_efgh_d2 + dist_modifier_diag );
                  )

         #endif

         #ifdef SECONDARY_ORTHOGONAL_RAYS

            DO_X1 ( PVEC dist_abcd_v ); DO_X1 ( PVEC dist_abcd_h );
            DO_X8 ( PVEC dist_efgh_v ); DO_X8 ( PVEC dist_efgh_h );

            DO_X1 ( dist_abcd_v.x = length ( ppos_a*star_hv_dfac.xy ); )
            DO_X1 ( dist_abcd_h.x = length ( ppos_a*star_hv_dfac.yx ); )

            DO_X2 ( dist_abcd_v.y = length ( ppos_b*star_hv_dfac.xy ); )
            DO_X2 ( dist_abcd_h.y = length ( ppos_b*star_hv_dfac.yx ); )

            DO_X4 ( dist_abcd_v.z = length ( ppos_c*star_hv_dfac.xy ); )
            DO_X4 ( dist_abcd_h.z = length ( ppos_c*star_hv_dfac.yx ); )

            DO_X4 ( dist_abcd_v.w = length ( ppos_d*star_hv_dfac.xy ); )
            DO_X4 ( dist_abcd_h.w = length ( ppos_d*star_hv_dfac.yx ); )

            DO_X8 ( dist_efgh_v.x = length ( ppos_e*star_hv_dfac.xy ); )
            DO_X8 ( dist_efgh_h.x = length ( ppos_e*star_hv_dfac.yx ); )

            DO_X8 ( dist_efgh_v.y = length ( ppos_f*star_hv_dfac.xy ); )
            DO_X8 ( dist_efgh_h.y = length ( ppos_f*star_hv_dfac.yx ); )

            DO_X8 ( dist_efgh_v.z = length ( ppos_g*star_hv_dfac.xy ); )
            DO_X8 ( dist_efgh_h.z = length ( ppos_g*star_hv_dfac.yx ); )

            DO_X8 ( dist_efgh_v.w = length ( ppos_h*star_hv_dfac.xy ); )
            DO_X8 ( dist_efgh_h.w = length ( ppos_h*star_hv_dfac.yx ); )

            DO_X1 ( pint_ad += star_hv_dist_ifactor / ( dist_abcd_h + dist_modifier_hv )
                             + star_hv_dist_ifactor / ( dist_abcd_v + dist_modifier_hv );
                  )

            DO_X8 ( pint_eh += star_hv_dist_ifactor / ( dist_efgh_h + dist_modifier_hv )
                             + star_hv_dist_ifactor / ( dist_efgh_v + dist_modifier_hv );
                  )

         #endif

         DO_X1 ( pint_ad = pow(pint_ad,PVEC(intensity_exponent)) * pint0 );
         DO_X8 ( pint_eh = pow(pint_eh,PVEC(intensity_exponent)) * pint0 );

         // Add this particle's contribution to the global colour.

         float hue = mix ( min_hue_shift,  max_hue_shift,  random_J(i,run_nr) ) + hue_time_factor*time3f;
         float sat = mix ( min_saturation, max_saturation, random_I(i,run_nr) ) * saturation_factor;

         DO_X1 ( PVEC sat_ad = min ( sat/pint_ad, 1.0 ); )
         DO_X8 ( PVEC sat_eh = min ( sat/pint_eh, 1.0 ); )

         DO_X1 ( PVEC val_ad = min ( pint_ad, 1.0); )
         DO_X8 ( PVEC val_eh = min ( pint_eh, 1.0); )

         vec3 qq = cos ( hue*tau + hsv2rgbXX ) - 1.0;

         DO_X1 ( pcol.r += dot ( val_ad, sat_ad * qq.r + 1.0 ); )
         DO_X1 ( pcol.g += dot ( val_ad, sat_ad * qq.g + 1.0 ); )
         DO_X1 ( pcol.b += dot ( val_ad, sat_ad * qq.b + 1.0 ); )

         DO_X8 ( pcol.r += dot ( val_eh, sat_eh * qq.r + 1.0 ); )
         DO_X8 ( pcol.g += dot ( val_eh, sat_eh * qq.g + 1.0 ); )
         DO_X8 ( pcol.b += dot ( val_eh, sat_eh * qq.b + 1.0 ); )

       }

   #endif // SECONDARY_PARTICLE_COUNT

   #ifdef SHOW_MAIN_PARTICLE

      // Get the main particle's position.

      vec2 ppos = scaled_mid_point;

      #ifdef MOVING_MAIN_PARTICLE
         ppos += vec2 ( harms ( scaled_x_amp, x_freq, x_phase, time2 ),
                        harms ( scaled_y_amp, y_freq, y_phase, time2 )
                     );
      #endif

      ppos = uv - ppos;

      // Generate either a blob or a four or eight pointed star.

      float dist = length ( ppos );
      float pint = inverse_distance_factor / ( dist + dist_modifier_radial );

      #ifdef MAIN_DIAGONAL_RAYS
         vec2 pposd = vec2 ( ppos.x+ppos.y, ppos.x-ppos.y );
         float dist_d1 = length ( pposd*star_diag_dfac.xy );
         float dist_d2 = length ( pposd*star_diag_dfac.yx );
         pint += stard_dist_ifactor / ( dist_d1 + dist_modifier_diag )
               + stard_dist_ifactor / ( dist_d2 + dist_modifier_diag );
      #endif
      
      #ifdef MAIN_ORTHOGONAL_RAYS
         float dist_v = length ( ppos*star_hv_dfac.xy );
         float dist_h = length ( ppos*star_hv_dfac.yx );
         pint += star_hv_dist_ifactor / ( dist_h + dist_modifier_hv )
               + star_hv_dist_ifactor / ( dist_v + dist_modifier_hv );
      #endif

      pint = pow(pint,mp_intensity_exponent) * mp_intensity_factors;

      // Add the main particle's contribution to the glbal colour.

      float sat = pow(pint,mp_sat_pow1)*mp_sat_pow2 + mp_sat_shift;
      float hue = hue_time_factor*time2             + mp_hue_shift;

      vec3 c = clamp ( vec3(hue,sat,pint), 0.0, 1.0 );

      pcol.rgb += c.z * ( c.y * (cos(c.x*tau+hsv2rgbXX) - 1.0 ) + 1.0 );

   #endif // SHOW_MAIN_PARTICLE

   // Make weak colours transparent and strong colours opaque.

   pcol.a = length(pcol.rgb);

   // Return the colour contributed by all the particles.

   return pcol;

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

void main ( void )
 {

   // Get co-ordinates with the source texture corresponding to the screen
   // pixel currently being processed. These are normalised and lie in the
   // range 0.0 to 1.0 - but see comments at the top of this shader.

   vec2 uv = gl_TexCoord[0].xy;

   // Get the base colour to use for the pixel.  If nothing else were to be
   // done by the shader then this would just display an unmodified clip.

   vec4 c1 = texture2D ( SourceImage, uv );

   // Get the colour contribution to the pixel from the swarm of particles.

   uv = uv*2.0 - 1.0;
   uv *= u_WindowSize/min(u_WindowSize.x,u_WindowSize.y);

   #ifdef MOTION_BLUR

      vec4 c2 = vec4(0.0);

      for (int s=0; s<MOTION_BLUR; s++)
       { c2 += drawParticles ( uv, float(s)*blur_time/float(MOTION_BLUR-1) );
       }

      c2 /= vec4(MOTION_BLUR);

   #else

      vec4 c2 = drawParticles ( uv, 0.0 );

   #endif

   // Make the base opacity of any transparent pixel slightly negative.  If
   // this is not done then the "background glow" from the  particles  will
   // cause the opacity to be non-zero everywhere within the area acted  on
   // by the shader and this will reveal the edges of the that  area  as  a
   // color discontinuity and a slightly discoloured rectangular region.

   if ( c1.a < 1.0 ) c1 = vec4(0.0,0.0,0.0,-0.25);

   // Combine the two colour contributions.

   gl_FragColor =  c1 + c2;

}

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
